/*****************************************************************************
 * VLC.cs
 *****************************************************************************
 * Copyright (C) 2005 Jon Lech Johansen <jon@nanocrew.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

using System;
using System.Runtime.InteropServices;

class VLC
{
    public enum Error
    {
        Success = -0,
        NoMem   = -1,
        Thread  = -2,
        Timeout = -3,

        NoMod   = -10,

        NoObj   = -20,
        BadObj  = -21,

        NoVar   = -30,
        BadVar  = -31,

        Exit    = -255,
        Generic = -666
    };

    enum Mode
    {
        Insert      = 0x01,
        Replace     = 0x02,
        Append      = 0x04,
        Go          = 0x08,
        CheckInsert = 0x10
    };

    enum Pos
    {
        End = -666
    };

    [DllImport("libvlc")]
    static extern int VLC_Create();
    [DllImport("libvlc")]
    static extern Error VLC_Init( int iVLC, int Argc, string [] Argv );
    [DllImport("libvlc")]
    static extern Error VLC_AddIntf( int iVLC, string Name,
                                     bool Block, bool Play );
    [DllImport("libvlc")]
    static extern Error VLC_Die( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_CleanUp( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_Destroy( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_AddTarget( int iVLC, string Target,
                                       string [] Options, int OptionsCount,
                                       int Mode, int Pos );
    [DllImport("libvlc")]
    static extern Error VLC_Play( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_Pause( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_Stop( int iVLC );
    [DllImport("libvlc")]
    static extern bool VLC_IsPlaying( int iVLC );
    [DllImport("libvlc")]
    static extern float VLC_PositionGet( int iVLC );
    [DllImport("libvlc")]
    static extern float VLC_PositionSet( int iVLC, float Pos );
    [DllImport("libvlc")]
    static extern int VLC_TimeGet( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_TimeSet( int iVLC, int Seconds, bool Relative );
    [DllImport("libvlc")]
    static extern int VLC_LengthGet( int iVLC );
    [DllImport("libvlc")]
    static extern float VLC_SpeedFaster( int iVLC );
    [DllImport("libvlc")]
    static extern float VLC_SpeedSlower( int iVLC );
    [DllImport("libvlc")]
    static extern int VLC_PlaylistIndex( int iVLC );
    [DllImport("libvlc")]
    static extern int VLC_PlaylistNumberOfItems( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_PlaylistNext( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_PlaylistPrev( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_PlaylistClear( int iVLC );
    [DllImport("libvlc")]
    static extern int VLC_VolumeSet( int iVLC, int Volume );
    [DllImport("libvlc")]
    static extern int VLC_VolumeGet( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_VolumeMute( int iVLC );
    [DllImport("libvlc")]
    static extern Error VLC_FullScreen( int iVLC );

    private int iVLC;

    public bool IsPlaying
    {
        get
        {
            return VLC_IsPlaying( iVLC );
        }
    }

    public VLC()
    {
        iVLC = VLC_Create();
        if( iVLC < 0 )
            throw new Exception( "VLC_Create failed" );
        Error err = VLC_Init( iVLC, 2, new string[] { "vlc", "--quiet" } );
        if( err != Error.Success )
        {
            VLC_Destroy( iVLC );
            throw new Exception( "VLC_Init failed" );
        }
    }

    ~VLC()
    {
        VLC_CleanUp( iVLC );
        VLC_Destroy( iVLC );
    }

    public Error AddTarget( string Target )
    {
        return VLC_AddTarget( iVLC, Target, null, 0,
                              (int)Mode.Append, (int)Pos.End );
    }

    public Error Play()
    {
        return VLC_Play( iVLC );
    }

    public Error Pause()
    {
        return VLC_Pause( iVLC );
    }

    public Error Stop()
    {
        return VLC_Stop( iVLC );
    }

    public Error Next()
    {
        return VLC_PlaylistNext( iVLC );
    }

    public Error Previous()
    {
        return VLC_PlaylistPrev( iVLC );
    }

    public Error ClearPlaylist()
    {
        return VLC_PlaylistClear( iVLC );
    }
}
